import React, { useState, createContext } from 'react';
import { HashRouter, Routes, Route, Navigate } from 'react-router-dom';
import Header from './components/Header';
import Footer from './components/Footer';
import Home from './pages/Home';
import Login from './pages/Login';
import Register from './pages/Register';
import GameLobby from './pages/GameLobby';
import TicTacToeGame from './pages/TicTacToeGame';
import Leaderboard from './pages/Leaderboard';
import Rules from './pages/Rules';
import GameSelection from './pages/GameSelection';
import LudoGame from './pages/LudoGame';
import ChessGame from './pages/ChessGame';
import BackgammonGame from './pages/BackgammonGame';

import type { User } from './types';
import { getUsers, saveUsers, getScores, saveScores } from './services/storageService';

interface AuthContextType {
  user: User | null;
  otherUser: User | null;
  isLoggedIn: boolean;
  setOtherUser: (user: User | null) => void;
  login: (username: string, pass: string) => boolean;
  logout: () => void;
  register: (username: string, pass: string) => { success: boolean, message: string };
}

export const AuthContext = createContext<AuthContextType | null>(null);

const App: React.FC = () => {
    const [user, setUser] = useState<User | null>(null);
    const [otherUser, setOtherUser] = useState<User | null>(null);
    const [isLoggedIn, setIsLoggedIn] = useState(false);

    const login = (username: string, pass: string): boolean => {
        const users = getUsers();
        const foundUser = users.find(u => u.username === username && u.password === pass);
        if (foundUser) {
            setUser(foundUser);
            setIsLoggedIn(true);
            return true;
        }
        return false;
    };

    const logout = () => {
        setUser(null);
        setOtherUser(null);
        setIsLoggedIn(false);
    };

    const register = (username: string, pass: string): { success: boolean, message: string } => {
        const users = getUsers();
        if (users.find(u => u.username === username)) {
            return { success: false, message: 'این نام کاربری قبلاً استفاده شده است.' };
        }
        const newUser: User = {
            id: `user-${Date.now()}`,
            username,
            password: pass
        };
        const newUsers = [...users, newUser];
        saveUsers(newUsers);

        // Also add new user to scores
        const scores = getScores();
        if (!scores.find(s => s.userId === newUser.id)) {
            const newScore = { userId: newUser.id, username: newUser.username, wins: 0, losses: 0, draws: 0, points: 0 };
            saveScores([...scores, newScore]);
        }

        setUser(newUser);
        setIsLoggedIn(true);
        return { success: true, message: 'ثبت نام موفقیت آمیز بود.' };
    };

    return (
        <AuthContext.Provider value={{ user, otherUser, isLoggedIn, setOtherUser, login, logout, register }}>
            <HashRouter>
                <div className="flex flex-col min-h-screen bg-gray-900">
                    <Header isLoggedIn={isLoggedIn} username={user?.username || null} onLogout={logout} />
                    <main className="flex-grow flex flex-col">
                        <Routes>
                            <Route path="/" element={<Home />} />
                            <Route path="/login" element={<Login />} />
                            <Route path="/register" element={<Register />} />
                            <Route path="/games" element={isLoggedIn ? <GameSelection /> : <Navigate to="/login" />} />
                            <Route path="/lobby/:gameType" element={isLoggedIn ? <GameLobby /> : <Navigate to="/login" />} />
                            <Route path="/game/tictactoe" element={isLoggedIn ? <TicTacToeGame /> : <Navigate to="/login" />} />
                            <Route path="/game/ludo" element={isLoggedIn ? <LudoGame /> : <Navigate to="/login" />} />
                            <Route path="/game/chess" element={isLoggedIn ? <ChessGame /> : <Navigate to="/login" />} />
                            <Route path="/game/backgammon" element={isLoggedIn ? <BackgammonGame /> : <Navigate to="/login" />} />
                            <Route path="/leaderboard" element={<Leaderboard />} />
                            <Route path="/rules" element={<Rules />} />
                        </Routes>
                    </main>
                    <Footer />
                </div>
            </HashRouter>
        </AuthContext.Provider>
    );
};

export default App;