import React, { useState, useEffect, useRef, FormEvent } from 'react';
import type { User } from '../types';

interface Message {
    senderId: string;
    senderName: string;
    text: string;
    timestamp: string;
}

interface ChatProps {
    currentUser: User;
    opponent: User;
}

const Chat: React.FC<ChatProps> = ({ currentUser, opponent }) => {
    const [messages, setMessages] = useState<Message[]>([]);
    const [inputValue, setInputValue] = useState('');
    const messagesEndRef = useRef<HTMLDivElement | null>(null);

    const scrollToBottom = () => {
        messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
    };

    useEffect(scrollToBottom, [messages]);
    
    useEffect(() => {
      // Add a welcome message when the component mounts
      setMessages([
        {
          senderId: 'system',
          senderName: 'سیستم',
          text: `چت با ${opponent.username} شروع شد.`,
          timestamp: new Date().toLocaleTimeString('fa-IR', { hour: '2-digit', minute: '2-digit' })
        }
      ]);
    }, [opponent.username]);

    const handleSendMessage = (e: FormEvent) => {
        e.preventDefault();
        if (inputValue.trim() === '') return;

        const newMessage: Message = {
            senderId: currentUser.id,
            senderName: currentUser.username,
            text: inputValue,
            timestamp: new Date().toLocaleTimeString('fa-IR', { hour: '2-digit', minute: '2-digit' })
        };

        const updatedMessages = [...messages, newMessage];
        setMessages(updatedMessages);
        setInputValue('');

        // Simulate opponent's response
        setTimeout(() => {
            const opponentResponse: Message = {
                senderId: opponent.id,
                senderName: opponent.username,
                text: opponent.id === 'bot-player' ? 'موفق باشی!' : 'جالبه!',
                timestamp: new Date().toLocaleTimeString('fa-IR', { hour: '2-digit', minute: '2-digit' })
            };
            setMessages(prevMessages => [...prevMessages, opponentResponse]);
        }, 1500 + Math.random() * 1000);
    };

    return (
        <div className="bg-gray-800 p-4 rounded-lg shadow-xl border border-gray-700 h-[400px] flex flex-col">
            <h3 className="text-xl font-bold mb-4 border-b border-gray-600 pb-2 text-white">چت بازی</h3>
            <div className="flex-grow overflow-y-auto pr-2 space-y-4">
                {messages.map((msg, index) => (
                    <div key={index} className={`flex flex-col ${msg.senderId === currentUser.id ? 'items-end' : 'items-start'}`}>
                        <div className={`rounded-lg px-3 py-2 max-w-[80%] ${msg.senderId === currentUser.id ? 'bg-indigo-600 text-white' : 'bg-gray-700 text-gray-200'}`}>
                            <p className="text-sm font-bold mb-1">{msg.senderId === currentUser.id ? 'شما' : msg.senderName}</p>
                            <p className="text-sm">{msg.text}</p>
                            <p className="text-xs text-right mt-1 opacity-60">{msg.timestamp}</p>
                        </div>
                    </div>
                ))}
                <div ref={messagesEndRef} />
            </div>
            <form onSubmit={handleSendMessage} className="mt-4 flex gap-2">
                <input
                    type="text"
                    value={inputValue}
                    onChange={(e) => setInputValue(e.target.value)}
                    placeholder="پیام خود را بنویسید..."
                    className="flex-grow px-4 py-2 bg-gray-700 border border-gray-600 rounded-md text-white focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 outline-none"
                />
                <button
                    type="submit"
                    className="py-2 px-4 bg-indigo-600 hover:bg-indigo-700 rounded-md text-white font-semibold transition-colors"
                >
                    ارسال
                </button>
            </form>
        </div>
    );
};

export default Chat;
