
import React, { useState, useEffect } from 'react';
import type { Score } from '../types';
import { getScores } from '../services/storageService';

const Leaderboard: React.FC = () => {
  const [scores, setScores] = useState<Score[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const allScores = getScores();
    const sortedScores = [...allScores].sort((a, b) => b.points - a.points);
    setScores(sortedScores);
    setLoading(false);
  }, []);

  const getRankColor = (rank: number) => {
    if (rank === 0) return 'text-yellow-400';
    if (rank === 1) return 'text-gray-300';
    if (rank === 2) return 'text-yellow-600';
    return 'text-gray-400';
  };

  const getRankIcon = (rank: number) => {
    if (rank === 0) return '🥇';
    if (rank === 1) return '🥈';
    if (rank === 2) return '🥉';
    return `${rank + 1}`;
  };

  if (loading) {
    return <div className="flex-grow text-center p-8">در حال بارگذاری جدول امتیازات...</div>;
  }

  return (
    <div className="flex-grow container mx-auto p-4 md:p-8">
      <div className="bg-gray-800 p-6 rounded-lg shadow-2xl border border-gray-700">
        <h1 className="text-4xl font-bold mb-6 text-center text-white bg-clip-text text-transparent bg-gradient-to-r from-purple-400 to-indigo-600">جدول برترین بازیکنان</h1>
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-700">
            <thead className="bg-gray-700/50">
              <tr>
                <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-300 uppercase tracking-wider">رتبه</th>
                <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-300 uppercase tracking-wider">نام بازیکن</th>
                <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-300 uppercase tracking-wider">امتیاز کل</th>
                <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-300 uppercase tracking-wider text-green-400">برد</th>
                <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-300 uppercase tracking-wider text-red-400">باخت</th>
                <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-300 uppercase tracking-wider text-yellow-400">مساوی</th>
              </tr>
            </thead>
            <tbody className="bg-gray-800 divide-y divide-gray-700">
              {scores.map((score, index) => (
                <tr key={score.userId} className="hover:bg-gray-700/50 transition-colors">
                  <td className={`px-6 py-4 whitespace-nowrap text-xl font-bold ${getRankColor(index)}`}>
                    {getRankIcon(index)}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-white">{score.username}</td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-300 font-bold">{score.points}</td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-green-400">{score.wins}</td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-red-400">{score.losses}</td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-yellow-400">{score.draws}</td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  );
};

export default Leaderboard;
