import React, { useState, useContext, useEffect, useCallback, useMemo } from 'react';
import { useNavigate, Link, useLocation } from 'react-router-dom';
import { AuthContext } from '../App';
import { useScoreUpdater } from '../hooks/useScoreUpdater';
import type { User } from '../types';
import { BOT_USER, LUDO_PLAYER_COLORS } from '../constants';
import Chat from '../components/Chat';

type PlayerKey = 'player1' | 'player2';
type PieceState = 'home' | 'active' | 'finished';

interface Piece {
  id: number;
  state: PieceState;
  position: number; // -1 for home, 0-51 for board path, 52-57 for home run
}

interface PlayerState {
  id: string;
  username: string;
  pieces: Piece[];
  color: { base: string, shadow: string, text: string };
  startPos: number;
}

// --- Game Constants ---
const PATH_LENGTH = 52;
const HOME_RUN_START = 52;
const FINISHED_POSITION = 58; // 52-57 are home run, 58 is finished
const START_POSITIONS: { [key in PlayerKey]: number } = { player1: 0, player2: 26 };
const SAFE_SPOTS = [0, 8, 13, 21, 26, 34, 39, 47];
const HOME_ENTRANCE: { [key in PlayerKey]: number } = { player1: 51, player2: 25 };

// --- Utility Functions ---
const getInitialPlayerState = (user: User, playerKey: PlayerKey): PlayerState => ({
  id: user.id,
  username: user.username,
  pieces: Array(4).fill(null).map((_, i) => ({ id: i, state: 'home', position: -1 })),
  color: LUDO_PLAYER_COLORS[playerKey],
  startPos: START_POSITIONS[playerKey],
});

// Maps a piece's relative position to its global path index
const getGlobalPathPosition = (player: PlayerState, piecePosition: number) => {
    return (player.startPos + piecePosition) % PATH_LENGTH;
};

const LudoGame: React.FC = () => {
    const auth = useContext(AuthContext);
    const navigate = useNavigate();
    const location = useLocation();
    const { updateScores } = useScoreUpdater();

    const opponent = useMemo(() => (location.state?.opponent as User | undefined) || auth?.otherUser, [location.state, auth?.otherUser]);
    const isBotGame = opponent?.id === BOT_USER.id;

    const [players, setPlayers] = useState<{ [key in PlayerKey]: PlayerState } | null>(null);
    const [gameState, setGameState] = useState<'ROLL' | 'MOVE' | 'GAMEOVER'>('ROLL');
    const [activePlayer, setActivePlayer] = useState<PlayerKey>('player1');
    const [diceValue, setDiceValue] = useState<number | null>(null);
    const [isRolling, setIsRolling] = useState(false);
    const [movablePieces, setMovablePieces] = useState<number[]>([]);
    const [winner, setWinner] = useState<PlayerKey | null>(null);
    const [gameMessage, setGameMessage] = useState('');

    const resetGame = useCallback(() => {
        if (auth?.user && opponent) {
            setPlayers({
                player1: getInitialPlayerState(auth.user, 'player1'),
                player2: getInitialPlayerState(opponent, 'player2'),
            });
            setActivePlayer('player1');
            setGameState('ROLL');
            setWinner(null);
            setDiceValue(null);
            setMovablePieces([]);
            setGameMessage('برای شروع تاس بیاندازید.');
        }
    }, [auth?.user, opponent]);

    useEffect(resetGame, [resetGame]);

    const switchTurn = useCallback(() => {
        const nextPlayer = activePlayer === 'player1' ? 'player2' : 'player1';
        setActivePlayer(nextPlayer);
        setGameState('ROLL');
        setDiceValue(null);
        if (nextPlayer === 'player1' || !isBotGame) {
            setGameMessage(`نوبت ${players![nextPlayer].username}. تاس بیاندازید.`);
        }
    }, [activePlayer, isBotGame, players]);

    const getMovablePieces = useCallback((playerKey: PlayerKey, die: number, currentPlayers: { [key in PlayerKey]: PlayerState }): number[] => {
        const player = currentPlayers[playerKey];
        return player.pieces
            .filter(piece => {
                if (piece.state === 'finished') return false;
                if (piece.state === 'home') return die === 6;
                
                const newPos = piece.position + die;
                return newPos < FINISHED_POSITION;
            })
            .map(piece => piece.id);
    }, []);

    const handleRollDice = useCallback(() => {
        if (gameState !== 'ROLL' || isRolling) return;
        setIsRolling(true);
        setDiceValue(null);
        setGameMessage('در حال انداختن تاس...');

        setTimeout(() => {
            const roll = Math.floor(Math.random() * 6) + 1;
            setDiceValue(roll);
            setIsRolling(false);
            
            const possibleMoves = getMovablePieces(activePlayer, roll, players!);
            if (possibleMoves.length > 0) {
                setMovablePieces(possibleMoves);
                setGameState('MOVE');
                setGameMessage('یک مهره برای حرکت انتخاب کنید.');
            } else {
                setGameMessage(`هیچ حرکتی با ${roll} ممکن نیست.`);
                setTimeout(switchTurn, 1500);
            }
        }, 1200);
    }, [gameState, isRolling, activePlayer, players, getMovablePieces, switchTurn]);
    
    const handlePieceMove = useCallback((pieceId: number) => {
        if (gameState !== 'MOVE' || !movablePieces.includes(pieceId) || !players || !diceValue) return;
        
        setMovablePieces([]);
        let newPlayers = JSON.parse(JSON.stringify(players)) as { [key in PlayerKey]: PlayerState };
        const player = newPlayers[activePlayer];
        const piece = player.pieces.find((p: Piece) => p.id === pieceId)!;
        
        let getsExtraTurn = diceValue === 6;

        if (piece.state === 'home' && diceValue === 6) {
            piece.state = 'active';
            piece.position = 0;
        } else if (piece.state === 'active') {
            const currentGlobalPos = getGlobalPathPosition(player, piece.position);
            
            // Check for entering home run
            if (currentGlobalPos <= HOME_ENTRANCE[activePlayer] && currentGlobalPos + diceValue > HOME_ENTRANCE[activePlayer]) {
                 const stepsAfterEntrance = (currentGlobalPos + diceValue) - HOME_ENTRANCE[activePlayer];
                 piece.position = HOME_RUN_START + stepsAfterEntrance - 1;
            } else {
                 piece.position += diceValue;
            }

            if (piece.position >= HOME_RUN_START + 6) {
                 piece.state = 'finished';
                 piece.position = FINISHED_POSITION;
                 getsExtraTurn = true;
            }
        }
        
        // Check for a kill
        if (piece.state === 'active' && piece.position < HOME_RUN_START) {
            const pieceGlobalPos = getGlobalPathPosition(player, piece.position);
            if (!SAFE_SPOTS.includes(pieceGlobalPos)) {
                const opponentKey = activePlayer === 'player1' ? 'player2' : 'player1';
                const opponent = newPlayers[opponentKey];
                
                opponent.pieces.forEach((oppPiece: Piece) => {
                    if (oppPiece.state === 'active') {
                        const oppGlobalPos = getGlobalPathPosition(opponent, oppPiece.position);
                        if (oppGlobalPos === pieceGlobalPos) {
                            oppPiece.state = 'home';
                            oppPiece.position = -1;
                            getsExtraTurn = true;
                            setGameMessage('مهره حریف زده شد! یک جایزه گرفتی.');
                        }
                    }
                });
            }
        }
        
        setPlayers(newPlayers);
        
        // Check for win
        const allFinished = newPlayers[activePlayer].pieces.every((p: Piece) => p.state === 'finished');
        if (allFinished) {
            setWinner(activePlayer);
            setGameState('GAMEOVER');
            if (auth.user && opponent) {
                updateScores(auth.user, opponent, activePlayer === 'player1' ? 'player1_wins' : 'player2_wins');
            }
            return;
        }

        if (getsExtraTurn) {
            setGameState('ROLL');
            setDiceValue(null);
            setGameMessage('یک جایزه گرفتی! دوباره تاس بیانداز.');
        } else {
            switchTurn();
        }

    }, [gameState, movablePieces, players, diceValue, activePlayer, auth.user, opponent, updateScores, switchTurn]);

    // Bot Logic
    useEffect(() => {
        if (isBotGame && activePlayer === 'player2' && gameState !== 'GAMEOVER' && !isRolling) {
            if (gameState === 'ROLL') {
                setTimeout(handleRollDice, 1500);
            } else if (gameState === 'MOVE' && movablePieces.length > 0) {
                 // Simple AI: just move the first possible piece
                 setTimeout(() => handlePieceMove(movablePieces[0]), 1500);
            }
        }
    }, [isBotGame, activePlayer, gameState, isRolling, movablePieces, handleRollDice, handlePieceMove]);

    if (!auth?.user || !opponent || !players) {
        return <div className="flex-grow flex items-center justify-center text-center p-8">در حال آماده‌سازی بازی...</div>;
    }
    
    const isMyTurn = activePlayer === 'player1' && gameState !== 'GAMEOVER';
    const player1 = players.player1;
    const player2 = players.player2;

    const renderBoard = () => (
      <div className="relative w-[90vmin] h-[90vmin] max-w-[500px] max-h-[500px] bg-gray-800 grid grid-cols-15 grid-rows-15">
        {/* Bases */}
        <div className="absolute top-0 left-0 w-[6_/_15_/_100%] h-[6_/_15_/_100%] bg-red-800/50 p-2"><div className="w-full h-full bg-red-800 grid grid-cols-2 grid-rows-2 p-1 gap-1">{player1.pieces.filter(p=>p.state==='home').map(p=><PieceComponent key={p.id} playerKey="player1" piece={p} onPieceClick={()=>{}} isMovable={false}/>)}</div></div>
        <div className="absolute bottom-right-0 bottom-0 right-0 w-[6_/_15_/_100%] h-[6_/_15_/_100%] bg-blue-800/50 p-2"><div className="w-full h-full bg-blue-800 grid grid-cols-2 grid-rows-2 p-1 gap-1">{player2.pieces.filter(p=>p.state==='home').map(p=><PieceComponent key={p.id} playerKey="player2" piece={p} onPieceClick={()=>{}} isMovable={false}/>)}</div></div>
        <div className="absolute top-0 right-0 w-[6_/_15_/_100%] h-[6_/_15_/_100%] bg-green-800/50"></div>
        <div className="absolute bottom-0 left-0 w-[6_/_15_/_100%] h-[6_/_15_/_100%] bg-yellow-800/50"></div>
        {/* Center */}
        <div className="col-start-7 col-span-3 row-start-7 row-span-3 bg-gray-900 flex items-center justify-center">
            <div className="w-0 h-0 border-l-[4vmin] border-l-transparent border-r-[4vmin] border-r-transparent border-b-[6vmin] border-b-red-600 -rotate-90"></div>
            <div className="w-0 h-0 border-l-[4vmin] border-l-transparent border-r-[4vmin] border-r-transparent border-b-[6vmin] border-b-blue-600 rotate-90"></div>
            <div className="w-0 h-0 border-l-[4vmin] border-l-transparent border-r-[4vmin] border-r-transparent border-b-[6vmin] border-b-green-600"></div>
            <div className="w-0 h-0 border-l-[4vmin] border-l-transparent border-r-[4vmin] border-r-transparent border-b-[6vmin] border-b-yellow-600 rotate-180"></div>
        </div>
        {/* Render active pieces */}
        {Object.entries(players).map(([playerKey, playerData]) => 
            playerData.pieces.filter(p => p.state !== 'home').map(p => 
                <PieceComponent key={`${playerKey}-${p.id}`} playerKey={playerKey as PlayerKey} piece={p} onPieceClick={handlePieceMove} isMovable={movablePieces.includes(p.id) && activePlayer === playerKey} />
            )
        )}
      </div>
    );
    
    const PieceComponent = ({ playerKey, piece, onPieceClick, isMovable }: { playerKey: PlayerKey, piece: Piece, onPieceClick: (id: number) => void, isMovable: boolean }) => {
        const player = players![playerKey];
        const color = player.color;
        let posStyle = {};
        
        // --- Path to Grid Coordinate Mapping ---
        const pathCoords = [
            {r:7,c:2},{r:7,c:3},{r:7,c:4},{r:7,c:5},{r:7,c:6}, {r:6,c:7},{r:5,c:7},{r:4,c:7},{r:3,c:7},{r:2,c:7},{r:1,c:7},
            {r:1,c:8}, {r:1,c:9},{r:2,c:9},{r:3,c:9},{r:4,c:9},{r:5,c:9},{r:6,c:9}, {r:7,c:10},{r:7,c:11},{r:7,c:12},{r:7,c:13},{r:7,c:14},{r:7,c:15},
            {r:8,c:15}, {r:9,c:15},{r:9,c:14},{r:9,c:13},{r:9,c:12},{r:9,c:11},{r:9,c:10}, {r:10,c:9},{r:11,c:9},{r:12,c:9},{r:13,c:9},{r:14,c:9},{r:15,c:9},
            {r:15,c:8}, {r:15,c:7},{r:14,c:7},{r:13,c:7},{r:12,c:7},{r:11,c:7},{r:10,c:7}, {r:9,c:6},{r:9,c:5},{r:9,c:4},{r:9,c:3},{r:9,c:2},{r:9,c:1},
            {r:8,c:1}
        ];
        const p1HomeRunCoords = [{r:2,c:8},{r:3,c:8},{r:4,c:8},{r:5,c:8},{r:6,c:8},{r:7,c:8}];
        const p2HomeRunCoords = [{r:14,c:8},{r:13,c:8},{r:12,c:8},{r:11,c:8},{r:10,c:8},{r:9,c:8}];

        if(piece.state === 'active'){
            const globalPos = getGlobalPathPosition(player, piece.position);
            const { r, c } = pathCoords[globalPos];
            posStyle = { gridRow: r, gridColumn: c };
        } else if (piece.state === 'finished' && piece.position >= HOME_RUN_START){
             const homeRunIndex = piece.position - HOME_RUN_START;
             const coords = playerKey === 'player1' ? p1HomeRunCoords : p2HomeRunCoords;
             if (homeRunIndex < coords.length) {
                const { r, c } = coords[homeRunIndex];
                posStyle = { gridRow: r, gridColumn: c };
             }
        }
        
        return (
            <div style={posStyle} className="relative flex items-center justify-center transition-all duration-500" onClick={() => onPieceClick(piece.id)}>
                <div className={`w-[70%] h-[70%] rounded-full border-2 border-black/50 ${color.base} ${isMovable ? `animate-pulse cursor-pointer ring-4 ring-white ${color.shadow}`: ''}`}></div>
            </div>
        )
    };

    const Dice = ({ value, isRolling, onClick } : { value: number | null, isRolling: boolean, onClick: ()=>void}) => (
        <button onClick={onClick} disabled={isRolling || !isMyTurn || gameState !== 'ROLL'} className="w-20 h-20 bg-gray-700 rounded-lg flex items-center justify-center text-5xl font-bold text-white transition-transform duration-300 transform perspective-1000 disabled:opacity-50 disabled:cursor-not-allowed hover:scale-105">
            {isRolling ? <div className="animate-spin text-4xl">🎲</div> : (value || '🎲')}
        </button>
    );

    return (
        <div className="flex-grow flex flex-col items-center justify-center p-2 sm:p-4 bg-gray-900 text-white">
            <h1 className="text-4xl font-bold mb-4">🎯 منچ</h1>
             <div className="flex flex-col lg:flex-row items-center lg:items-start gap-8 w-full max-w-7xl mx-auto">
                <div className="w-full lg:w-64 flex-shrink-0 order-2 lg:order-1">
                     <div className={`bg-gray-800 p-4 rounded-lg shadow-xl border-2 ${activePlayer === 'player1' ? 'border-red-500' : 'border-gray-700'}`}>
                        <h3 className={`text-xl font-bold mb-2 ${player1.color.text}`}>{player1.username} (شما)</h3>
                         <p>مهره تمام‌شده: {player1.pieces.filter(p=>p.state==='finished').length}</p>
                     </div>
                </div>
                 <div className="flex-grow flex flex-col items-center order-1 lg:order-2">
                     <div className="mb-4 text-center p-3 bg-gray-800 rounded-lg min-h-[5rem] w-full">
                        {gameState === 'GAMEOVER' ? (
                            <div className="text-2xl font-bold text-green-400">{players[winner!].username} برنده شد!</div>
                        ) : (
                            <h2 className="text-xl font-bold text-gray-300">{gameMessage}</h2>
                        )}
                     </div>
                    {renderBoard()}
                    <div className="mt-4">
                        <Dice value={diceValue} isRolling={isRolling} onClick={handleRollDice}/>
                    </div>
                    {gameState === 'GAMEOVER' && (
                        <div className="mt-6">
                             <button onClick={resetGame} className="bg-indigo-600 hover:bg-indigo-700 text-white font-bold py-3 px-8 rounded-full text-lg transition-transform transform hover:scale-105 shadow-lg">
                                بازی مجدد
                            </button>
                        </div>
                    )}
                 </div>
                <div className="w-full lg:w-64 flex-shrink-0 order-3 lg:order-3 space-y-4">
                     <div className={`bg-gray-800 p-4 rounded-lg shadow-xl border-2 ${activePlayer === 'player2' ? 'border-blue-500' : 'border-gray-700'}`}>
                         <h3 className={`text-xl font-bold mb-2 ${player2.color.text}`}>{player2.username}</h3>
                         <p>مهره تمام‌شده: {player2.pieces.filter(p=>p.state==='finished').length}</p>
                     </div>
                     <Chat currentUser={auth.user} opponent={opponent} />
                </div>
             </div>
        </div>
    );
};

export default LudoGame;
